/* main.c - ADC SW Scan example for S32R274 */
/* Description:  Converts inputs ADC1_AN_1 using scan mode (continuous) */
/* Rev 1.0 Aug 13 2018 D Chung - production version */
/* Copyright NXP Semiconductor, Inc 2018 All rights reserved. */

/*******************************************************************************
* NXP Semiconductor Inc.
* (c) Copyright 2018 NXP Semiconductor, Inc.
* ALL RIGHTS RESERVED.
********************************************************************************
Services performed by NXP in this matter are performed AS IS and without
any warranty. CUSTOMER retains the final decision relative to the total design
and functionality of the end product. NXP neither guarantees nor will be
held liable by CUSTOMER for the success of this project.
NXP DISCLAIMS ALL WARRANTIES, EXPRESSED, IMPLIED OR STATUTORY INCLUDING,
BUT NOT LIMITED TO, IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR
A PARTICULAR PURPOSE ON ANY HARDWARE, SOFTWARE ORE ADVISE SUPPLIED TO THE PROJECT
BY NXP, AND OR NAY PRODUCT RESULTING FROM NXP SERVICES. IN NO EVENT
SHALL NXP BE LIABLE FOR INCIDENTAL OR CONSEQUENTIAL DAMAGES ARISING OUT OF
THIS AGREEMENT.

CUSTOMER agrees to hold NXP harmless against any and all claims demands or
actions by anyone on account of any damage, or injury, whether commercial,
contractual, or tortuous, rising directly or indirectly as a result of the advise
or assistance supplied CUSTOMER in connection with product, services or goods
supplied under this Agreement.

Copyright 2018 NXP.  This software is owned or controlled by NXP and may only
be used strictly in accordance with the applicable license terms.  By expressly
accepting such terms or by downloading, installing, activating and/or otherwise
using the software, you are agreeing that you have read, and that you agree to
comply with and are bound by, such license terms.  If you do not agree to be
bound by the applicable license terms, then you may not retain, install, activate
or otherwise use the software.
********************************************************************************
* File              main.c
* Owner             David Chung
* Version           1.0
* Date              Aug-13-2018
* Classification    General Business Information
* Brief             Tests S32R274 ADC. LEDs turn on/off based on ADC result.
********************************************************************************
* Detailed Description:
* Application performs basic ADC initialization.  ADC scans ADC1_AN_1 in
* continuous mode.  The user LEDs turn on based on ADC result.
*
* ------------------------------------------------------------------------------
* Test HW:         S32R274RRUEVB+MPC57xxMB
* MCU:             S32R274
* Terminal:        None
* Fsys:            160 MHz
* Debugger:        USB Multilink
* Target:          FLASH
* EVB connection:  Motherboard connections:
* 					Connect potentiometer (RV1) to its pin. Short J53.
* 					PA0 (P8.1) to D2 (P7.1), PA1 (P8.2) to D3 (P7.2),
* 					PA2 (P8.3) to D4 (P7.3), PA3 (P8.4) to D5 (P7.4),
* 					and Potentiometer pin (P9.1) to	ADC1_AN_1 (P9.15)
*
********************************************************************************
Revision History:
Version  Date         Author  			Description of Changes
1.0      Aug-13-2018  David Chung	  	Initial version

*******************************************************************************/

#include "derivative.h" /* include peripheral declarations */
#include "project.h"
#include "mode_entry.h"
#include "adc.h"

void peri_clock_gating(void);  /* Configure gating/enabling peri. clocks */
void LED_Config(void);   /* Assign LED ports on Freescale EVBs as GPIO outputs */
void update_LEDs(void); /* Update LEDs with scaled chan 9 result */
extern uint16_t Result[3];     /* ADC channel conversion results */

#define KEY_VALUE1 0x5AF0ul
#define KEY_VALUE2 0xA50Ful

extern void xcptn_xmpl(void);

void hw_init(void)
{
#if defined(TURN_ON_CPU1) || defined(TURN_ON_CPU2)
	uint32_t mctl = MC_ME.MCTL.R;
#endif
#if defined(TURN_ON_CPU1)
	/* enable core 1 in all modes */
	MC_ME.CCTL2.R = 0x00FE;
	/* Set Start address for core 1: Will reset and start */
#if defined(START_FROM_FLASH)
    MC_ME.CADDR2.R = 0x1080000 | 0x1;
#else
    MC_ME.CADDR2.R = 0x4006a800 | 0x1;
#endif /* defined(START_FROM_FLASH) */
	
#endif	
#if defined(TURN_ON_CPU2)
	/* enable core 2 in all modes */
	MC_ME.CCTL3.R = 0x00FE;
	/* Set Start address for core 2: Will reset and start */
#if defined(START_FROM_FLASH)
    MC_ME.CADDR3.R = 0x1100000 | 0x1;
#else
    MC_ME.CADDR3.R = 0x400d5000 | 0x1;
#endif /* defined(START_FROM_FLASH) */
	
#endif
#if defined(TURN_ON_CPU1) || defined(TURN_ON_CPU2)
	MC_ME.MCTL.R = (mctl & 0xffff0000ul) | KEY_VALUE1;
	MC_ME.MCTL.R =  mctl; /* key value 2 always from MCTL */
#endif
}

__attribute__ ((section(".text")))
int main(void)
{
	int counter = 0;

	xcptn_xmpl ();              /* Configure and Enable Interrupts */

	peri_clock_gating();    /* Configure gating/enabling peri. clocks for modes*/
	system160mhz();         /* sysclk=160MHz, dividers configured, mode trans*/

	LED_Config();           /* Assign LED ports on Freescale LED as GPIO outputs*/

	CGM.AC0_SC.B.SELCTL = 0b010; //Select PLL0_PHI_CLK as source of ADC_CLK

	CGM.AC0_DC2.R = 0x80020000; //Enable ADC_CLK and divide PLL0_PHI_CLK source by 3 (i.e. 4+1). 240MHz/3 = 80 MHz

	ADC1_PadConfig_ChanSelect();  /* Configure ADC pads & select scan channels */
	ADC1_Calibration();     /* Calibrate to compensate for variations */
	ADC1_Init();            /* Initialize ADC1 module & start normal scan mode */

	for(;;) {	   
	   	counter++;
		if (ADC_1.ISR.B.ECH) { /* If normal scan channels finished converting */
		  ADC1_Read_Chan();    /* Read conversion results */
		  update_LEDs();       /* Update LEDs with scaled chan 1 result */
		  ADC_1.ISR.R = 0x00000001; /* Clear End of CHain (ECH) status bit */
		}
	}

	return 0;
}

void peri_clock_gating (void) {
  MC_ME.RUN_PC[0].R = 0x00000000;  /* gate off clock for all RUN modes */
  MC_ME.RUN_PC[1].R = 0x000000FE;  /* enable peri clock for all RUN modes */
                                   /* Note: RUN_PC0 is default cfg for PCTLs */
  MC_ME.PCTL[126].B.RUN_CFG = 0x1; //Point ADC1 to RUN_PC[1]
}

void LED_Config(void) {        /* Assign LED ports as GPIO outputs */

	/* Configure the user LEDs on S32R274RRUEVB+MPC57xxMB */
	SIUL2.MSCR[0].B.OBE = 1;
	SIUL2.MSCR[1].B.OBE = 1;
	SIUL2.MSCR[2].B.OBE = 1;
	SIUL2.MSCR[3].B.OBE = 1;

	/* Turn off LEDs. LEDs are active low */
	SIUL2.GPDO[0].B.PDO_4n = 1;
	SIUL2.GPDO[0].B.PDO_4n1 = 1;
	SIUL2.GPDO[0].B.PDO_4n2 = 1;
	SIUL2.GPDO[0].B.PDO_4n3 = 1;

}

void update_LEDs(void) {     /* Update LEDs with scaled chan 1 result  */
                             /* If Result bit is 0, then LED is turned ON */
                             /* If Result bit is 1, then LED is turned OFF */
	SIUL2.GPDO[0].B.PDO_4n  = (Result[0] & 0x0800)>>11; /* D2: scaled ch 1 LSB */
	SIUL2.GPDO[0].B.PDO_4n1  = (Result[0] & 0x0400)>>10; /* D3 */
	SIUL2.GPDO[0].B.PDO_4n2 = (Result[0] & 0x0200)>>9;  /* D4 */
	SIUL2.GPDO[0].B.PDO_4n3 = (Result[0] & 0x0100)>>8;  /* D5 */

}


